<?php
/*
Plugin Name: bbPress Moderation
Description: Moderate bbPress topics and replies
Author: Ian Haycox
Version: 1.2
Author URI: http://ianhaycox.com


 Copyright Ian Haycox, 2011 (email : ian.haycox@gmail.com)

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details. 

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

class bbPressModeration {
	
	const TD = 'bbpressmoderation'; 

	/**
	 * Construct
	 * 
	 * Setup filters and actions for bbPress
	 */
	function __construct() {
		add_filter('bbp_new_topic_pre_insert', array($this, 'pre_insert'));
		add_filter('bbp_new_reply_pre_insert', array($this, 'pre_insert'));

		add_filter('bbp_new_topic_redirect_to', array($this, 'redirect_to'), 10 , 2);
		
		add_filter('bbp_has_topics_query', array($this, 'query'));
		add_filter('bbp_has_replies_query', array($this, 'query'));

		add_filter('bbp_get_topic_permalink', array($this, 'permalink'), 10, 2);
		add_filter('bbp_get_reply_permalink', array($this, 'permalink'), 10, 2);

		add_filter('bbp_get_topic_title', array($this, 'title'));

		add_filter('bbp_get_reply_content', array($this, 'content'));

		add_filter('bbp_current_user_can_publish_replies', array($this, 'can_reply'));
		
		add_action('bbp_new_topic', array($this, 'new_topic'), 10, 4);
		add_action('bbp_new_reply', array($this, 'new_reply'), 10, 5);
		
		load_plugin_textdomain(self::TD, false, dirname( plugin_basename( __FILE__ ) ) . '/languages');
		
		if (is_admin()) {
			// Activation/deactivation functions
			register_activation_hook(__FILE__, array(&$this, 'activate'));
			register_deactivation_hook(__FILE__, array(&$this, 'deactivate'));
		
			// Register an uninstall hook to automatically remove options
			register_uninstall_hook(__FILE__, array('bbPressModeration', 'deinstall') );
			
			add_action('admin_init', array($this, 'admin_init' ));
			add_action('admin_menu', array($this, 'admin_menu' ));
			add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), array(&$this, 'plugin_action_links'));
		}
	}

	/**
	 * Activate
	 * @return boolean
	 */
	function activate() {
		// Notify admin 
		add_option(self::TD . 'notify', 1);
		add_option(self::TD . 'always_approve', 1);
		add_option(self::TD . 'previously_approved', 1);
		return true;
	}
	
	/**
	 * Deactivate
	 * @return boolean
	 */
	function deactivate() {
		return true;
	}
	
	/**
	 * Tidy up deleted plugin by removing options
	 */
	static function deinstall() {
		delete_option(self::TD . 'notify');
		delete_option(self::TD . 'always_approve');
		delete_option(self::TD . 'previously_approved');
		return true;
	}
	
	/**
	 * Before inserting a new topic/reply mark
	 * this as 'pending' depending on settings
	 * 
	 * @param array $data - new topic/reply data
	 */
	function pre_insert($data) {
		global $wpdb;
		
		// Pointless moderating a post that the current user can approve
		if (current_user_can('moderate')) return $data;
		
		if ($data['post_author'] == 0) {
			// Anon user - check if need to moderate
		 	if (get_option(self::TD . 'always_approve')) {
				$data['post_status'] = 'pending';
			}
		} else {
			// Registered user
			if (get_option(self::TD . 'previously_approved')) {
				// Check if user already published 
				$sql = $wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_author = %d AND post_type IN ('topic','reply') AND post_status = 'publish'", $data['post_author']);
				$count = $wpdb->get_var($sql);
				if (!$count) {
					// Anon or User never published topic/reply so mark as pending.
					$data['post_status'] = 'pending';
				}
			} else {
				$data['post_status'] = 'pending';
			}
		}
		
		return $data;
	}

	/**
	 * When querying for topics/replies include those
	 * marked as pending otherwise we never see them
	 * 
	 * @param array $bbp topic/reply query data
	 */
	function query($bbp) {
		if (!bbp_is_query_name( 'bbp_widget' )) {
			$bbp['post_status'] .= ',pending';
		}
		return $bbp;
	}

	/**
	 * Trash the permalink for pending topics/replies
	 * Would be nice if we could remove the link entirely
	 * but the filter is a bit too late
	 * 
	 * @param string $permalink - topic or reply permalink
	 * @param int $topic_id - topic or reply ID
	 */
	function permalink($permalink, $topic_id) {
		global $post;
		
		if (!current_user_can('moderate') && $post && $post->post_status == 'pending') {
			return '#';  // Fudge url to prevent viewing
		}
		
		return $permalink;
	}

	/**
	 * Alter pending topic title to indicate it
	 * is awaiting moderation
	 * 
	 * @param string $title - the title
	 * @return string New title
	 */
	function title($title) {
		global $post;

		if ($post && $post->post_status == 'pending') {
			return $title . ' ' . __('(Awaiting moderation)', self::TD);
		}

		return $title;
	}

	/**
	 * Hide content for pending replies
	 * 
	 * @param string $content - the content
	 * @return string - New content
	 */
	function content($content) {
		global $post;

		if ($post && $post->post_status == 'pending') {
			if (current_user_can('moderate')) {
				// Admin can see body
				return __('(Awaiting moderation)', self::TD) . '<br />' . $content;
			} else {
				return __('(Awaiting moderation)', self::TD);
			}
		}

		return $content;
	}

	/**
	 * Check if newly created topic is published and
	 * disable replies until it is.
	 * 
	 * @param boolean $retval - Indicator if user can reply
	 * @return boolean - true can reply
	 */
	function can_reply($retval) {
		if (!$retval) return $retval;

		$topic_id = bbp_get_topic_id();
		
		return ('publish' == bbp_get_topic_status($topic_id));
	}

	/**
	 * Notify admin of new reply with pending status
	 * 
	 * @param int $reply_id
	 * @param int $topic_id
	 * @param int $forum_id
	 * @param boolean $anonymous_data
	 * @param int $reply_author
	 */
	function new_reply($reply_id = 0, $topic_id = 0, $forum_id = 0, $anonymous_data = false, $reply_author = 0) {
		$reply_id = bbp_get_reply_id( $reply_id );
		
		$status = bbp_get_reply_status($reply_id);
		
		if ($status == 'pending') {
			$this->notify_admin($reply_id);
		}
	}
	
	/**
	 * Notify admin of new topic with pending status
	 * 
	 * @param int $topic_id
	 * @param int $forum_id
	 * @param boolean $anonymous_data
	 * @param int $reply_author
	 */
	function new_topic($topic_id = 0, $forum_id = 0, $anonymous_data = false, $topic_author = 0) {
		$topic_id = bbp_get_topic_id( $topic_id );
		
		$status = bbp_get_topic_status($topic_id);
		
		if ($status == 'pending') {
			$this->notify_admin($topic_id);
		}
	}
	
	/**
	 * Alert admin of pending topic/reply
	 */
	function notify_admin($post_id) {
		
		if (get_option(self::TD . 'notify')) {
			
			$blogname = wp_specialchars_decode(get_option('blogname'), ENT_QUOTES);
			$blogurl = get_option('siteurl');
			$message  = sprintf(__('New topic/reply awaiting moderation on your site %s: %s', self::TD), $blogname, $blogurl) . "\r\n\r\n";

			$message .= $post_id . "\r\n\r\n";
				
			/* Add body of topic/reply to email */
			$post = get_post($post_id);
			
			if ($post) {
				$message .= get_permalink($post->ID) . "\r\n\r\n";
				$message .= $post->post_content . "\r\n\r\n";
			}
			
			@wp_mail(get_option('admin_email'), sprintf(__('[%s] bbPress Moderation', self::TD), $blogname), $message);
		}
	}
	
	/**
	 * Show pending counts for topics/replies and
	 * add plugin options page to Settings
	 */
	function admin_menu() {
		global $menu;
		global $wpdb;
		
		add_options_page(__('bbPress Moderation', self::TD), __('bbPress Moderation', self::TD), 
								'manage_options', self::TD, array($this, 'options'));
		
		/*
		 * Are there any pending items ?
		 */
		$sql = "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'topic' AND post_status = 'pending'";
		$topic_count = $wpdb->get_var($sql);
		$sql = "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'reply' AND post_status = 'pending'";
		$reply_count = $wpdb->get_var($sql);
		
		if ($reply_count || $topic_count) {
			/*
			 * Have a quick butchers at the menu structure
			 * looking for topics and replies and tack
			 * on the pending bubble count. The 5th item seems
			 * to be the class name
			 * 
			 * Bit hacky but seems to work 
			 */
			foreach ($menu as $key=>$item) {
				if ($topic_count && isset($item[5]) && $item[5] == 'menu-posts-topic') {
					$bubble = '<span class="awaiting-mod count-'.$topic_count.'"><span class="pending-count">'.number_format_i18n($topic_count) .'</span></span>';
					$menu[$key][0] .= $bubble;
				}
				if ($reply_count && isset($item[5]) && $item[5] == 'menu-posts-reply') {
					$bubble = '<span class="awaiting-mod count-'.$reply_count.'"><span class="pending-count">'.number_format_i18n($reply_count) .'</span></span>';
					$menu[$key][0] .= $bubble;
				}
			}
		}
	}
	
	/**
	 * Check if we need to redirect this pending topic
	 * back to the forum list of topics
	 * 
	 * @param string $redirect_url
	 * @param string $redirect_to
	 * @return string
	 */
	function redirect_to($redirect_url, $redirect_to) {
		
		if (!empty($redirect_url)) {
			$query = parse_url($redirect_url, PHP_URL_QUERY);
			$args = wp_parse_args($query);
			
			if (isset($args['p'])) {
				$topic_id = bbp_get_topic_id($args['p']);
				
				if ('pending' == bbp_get_topic_status($topic_id)) {
					return $_SERVER['HTTP_REFERER'];
				}
			}
		}
		
		return $redirect_url;
	}
	
	/**
	 * Add settings link to plugin admin page
	 * 
	 * @param array $links - plugin links
	 * @return array of links
	 */
	function plugin_action_links($links) {
		$settings_link = '<a href="admin.php?page=bbpressmoderation">'.__('Settings', self::TD).'</a>';
		$links[] = $settings_link;
		return $links;
	}
	
	/**
	 * Register our options
	 */
	function admin_init() {
		register_setting( self::TD.'option-group', self::TD.'notify');
		register_setting( self::TD.'option-group', self::TD.'always_approve');
		register_setting( self::TD.'option-group', self::TD.'previously_approved');
	}
	
	/**
	 * Plugin settings page for various options
	 */
	function options() {
		?>
		
		<div class="wrap">
		
		<div id="icon-options-general" class="icon32"><br/></div>
		<h2><?php _e('bbPress Moderation settings', self::TD); ?></h2>
		
		<form method="post" action="options.php">
		
		<?php settings_fields( self::TD.'option-group' );?>
		
		<table class="form-table">
		<tr valign="top">
		<th scope="row"><?php _e('E-mail me whenever', self::TD); ?></th>
		<td>
			<input type="checkbox" id="<?php echo self::TD; ?>notify" name="<?php echo self::TD; ?>notify" value="1" <?php echo (get_option(self::TD.'notify', '') ? ' checked ' : ''); ?>" />
			<label for="<?php echo self::TD; ?>notify"><?php _e('A topic or reply is held for moderation', self::TD); ?></label>
		</td>
		</tr>
		
		<tr valign="top">
		<th scope="row"><?php _e('Do not moderate', self::TD); ?></th>
		<td>
			<input type="checkbox" id="<?php echo self::TD; ?>previously_approved" name="<?php echo self::TD; ?>previously_approved" value="1" <?php echo (get_option(self::TD.'previously_approved', '') ? ' checked ' : ''); ?>" />
			<label for="<?php echo self::TD; ?>previously_approved"><?php _e('A topic or reply by a previously approved author', self::TD); ?></label>
		</td>
		</tr>

		<tr valign="top">
		<th scope="row"><?php _e('Anonymous topics and replies', self::TD); ?></th>
		<td>
			<input type="checkbox" id="<?php echo self::TD; ?>always_approve" name="<?php echo self::TD; ?>always_approve" value="1" <?php echo (get_option(self::TD.'always_approve', '') ? ' checked ' : ''); ?>" />
			<label for="<?php echo self::TD; ?>always_approve"><?php _e('Always moderate', self::TD); ?></label>
		</td>
		</tr>

		</table>
		
		
		<p class="submit">
		<input type="submit" class="button" value="<?php _e('Save Changes', self::TD); ?>" />
		</p>
		</form>
		
		<p>
		<?php _e('If you have any problems, please read the <a href="http://wordpress.org/extend/plugins/bbpressmoderation/faq/">FAQ</a> and if necessary <a href="http://ianhaycox.com/contact">contact me</a>.', self::TD); ?>
		</p>
		<p>
		<?php _e('If you like this plugin please consider <a href="http://ianhaycox.com/donate">donating</a> to fund future development. Thank you.', self::TD); ?>
		</p>
		
		
		</div>

<?php 
	}
}

$bbpressmoderation = new bbPressModeration();
